#!/usr/bin/env python3
"""
check_shortcuts.py
──────────────────
Recursively scans ../texts/list_of_shortcuts (by default) for files that contain
the four shortcut lines

    <ctrl>T
    <ctrl>minus
    <ctrl>plus
    <ctrl>B

( each line must start with an actual TAB character, *not* spaces ) and checks
whether

 • the translations for <ctrl>T and <ctrl>minus     are *identical*
 • the translations for <ctrl>plus and <ctrl>B      are *identical*

If either pair is identical, the script prints a short report naming the file
and the offending pair.

Usage
-----

    $ python3 check_shortcuts.py                     # scan ../texts/list_of_shortcuts
    $ python3 check_shortcuts.py path/to/dir         # scan a specific directory
"""
from __future__ import annotations

import argparse
import os
import re
import sys
from pathlib import Path

# ──────────────────────────────────────────────────────────────────────────────
# 1. Command–line arguments
# ──────────────────────────────────────────────────────────────────────────────
parser = argparse.ArgumentParser(description="Detect identical translations in CTRL shortcut lines.")
parser.add_argument(
    "root",
    nargs="?",
    default=str(Path(__file__).parent / "../txts/list_of_shortcuts"),
    help="Directory to scan (default: ../txts/list_of_shortcuts relative to script)",
)
args = parser.parse_args()
root: Path = Path(args.root).resolve()

# ──────────────────────────────────────────────────────────────────────────────
# 2. Pre-compile the regular expressions
# ──────────────────────────────────────────────────────────────────────────────
TOKEN_PATTERNS: dict[str, re.Pattern[str]] = {
    "<ctrl>T": re.compile(r"^\t<ctrl>T\s*(?:\|\s*)?(.*)$"),
    "<ctrl>minus": re.compile(r"^\t<ctrl>minus\s*(?:\|\s*)?(.*)$"),
    "<ctrl>plus": re.compile(r"^\t<ctrl>plus\s*(?:\|\s*)?(.*)$"),
    "<ctrl>B": re.compile(r"^\t<ctrl>B\s*(?:\|\s*)?(.*)$"),
}

# ──────────────────────────────────────────────────────────────────────────────
# 3. Helpers
# ──────────────────────────────────────────────────────────────────────────────
def collect_translations(path: Path) -> dict[str, str]:
    found: dict[str, str] = {}
    try:
        with path.open(encoding="utf-8", errors="ignore") as fh:
            for line in fh:
                for token, pat in TOKEN_PATTERNS.items():
                    if token not in found:
                        m = pat.match(line)
                        if m:
                            found[token] = m.group(1).strip()
                if len(found) == 4:
                    return found
    except (UnicodeDecodeError, OSError):
        pass
    return found


def report(path: Path, dup_T_minus: bool, dup_plus_B: bool, trans: dict[str, str]) -> None:
    print(f"\n{path}")
    if dup_T_minus:
        print(f"  • identical translations for <ctrl>T & <ctrl>minus : “{trans['<ctrl>T']}”")
    if dup_plus_B:
        print(f"  • identical translations for <ctrl>plus & <ctrl>B   : “{trans['<ctrl>plus']}”")


# ──────────────────────────────────────────────────────────────────────────────
# 4. Walk the tree
# ──────────────────────────────────────────────────────────────────────────────
duplicate_count = 0

for dirpath, _dirnames, filenames in os.walk(root):
    for name in filenames:
        file_path = Path(dirpath, name)
        translations = collect_translations(file_path)

        if {"<ctrl>T", "<ctrl>minus"}.issubset(translations) or {
            "<ctrl>plus", "<ctrl>B"
        }.issubset(translations):
            dup_pair_1 = translations.get("<ctrl>T") == translations.get("<ctrl>minus")
            dup_pair_2 = translations.get("<ctrl>plus") == translations.get("<ctrl>B")

            if dup_pair_1 or dup_pair_2:
                duplicate_count += 1
                report(file_path, dup_pair_1, dup_pair_2, translations)

# ──────────────────────────────────────────────────────────────────────────────
# 5. Final summary
# ──────────────────────────────────────────────────────────────────────────────
if duplicate_count == 0:
    print("√ No identical shortcut translations found.")
else:
    print(f"\nFound {duplicate_count} file(s) with identical shortcut translations.")

