# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023,2024 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import mobius
from . import decoder_done_list_dat
from . import decoder_history_list_dat
from . import decoder_task_list_dat

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
APP_ID = "itubego"
APP_NAME = "iTubeGo"


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief iTubeGo Profile class
# @author Eduardo Aguiar
# @see https://itubego.com/
#
# iTubeGo artifacts are located at folder AppData/Local/iTubeGo
#
# Files:
#   done_list.dat           Downloaded files
#   history_list.dat        Typed URls (autocomplete)
#   private_list.dat        ???
#   task_list.dat           Pending downloads
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Profile(object):

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Initialize object
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __init__(self, folder, item, username):
        self.__folder = folder
        self.__item = item

        # set profile attributes
        self.name = username
        self.path = folder.path
        self.folder = folder
        self.metadata = mobius.pod.map()
        self.username = username
        self.app_id = APP_ID
        self.app_name = APP_NAME

        # set data attributes
        self.__history_loaded = False
        self.__downloaded_files_loaded = False
        self.__history_entries = []
        self.__downloaded_files = []

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Get history list entries
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def get_history_entries(self):
        self.__load_history_entries()
        return self.__history_entries

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Get downloaded files
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def get_downloaded_files(self):
        self.__load_downloaded_files()
        return self.__downloaded_files

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Load history list entries (from history_list.dat)
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __load_history_entries(self):
        if self.__history_loaded:
            return

        for child in self.folder.get_children():
            if child.name.lower() == 'history_list.dat' and not child.is_reallocated():
                self.__history_entries += decoder_history_list_dat.decode(child)

        self.__history_loaded = True

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Load downloaded files (from done_list.dat and task_list.dat)
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __load_downloaded_files(self):
        if self.__downloaded_files_loaded:
            return

        for child in self.folder.get_children():
            if child.name.lower() == 'done_list.dat' and not child.is_reallocated():
                self.__downloaded_files += decoder_done_list_dat.decode(child)

            elif child.name.lower() == 'task_list.dat' and not child.is_reallocated():
                self.__downloaded_files += decoder_task_list_dat.decode(child)

        self.__downloaded_files_loaded = True
